function [press,err] = plspress(x,y,useR,leaveR)
%PLSPRESS  Average leave-out-and-prediction error (PRESS in concentration
%          units) achieved from a PLS regression with mean-centered data.
%
%          PRESS = PLSPRESS(X,Y,USER)         or
%          PRESS = PLSPRESS(X,Y,USER,LEAVER)
%
%          Given regression matrix X and regressand vector Y, PRESS
%          provides the average PLS-prediction error by using the
%          leave-out-and-predict strategy for the USER first PLS ranks.
%          If LEAVER(1) is given, PRESS deletes standards in consecutive
%          groups of LEAVER standards (not just one).
%
%          With size(PRESS)=[2,useR+1], the precise form of the output is:
%
%                     PRESS
%             PRESS =       
%                      cnt        (to detect pre-mature PLS stopping)
%
%          the first column meaning 'prediction by average value'.

%          R. Marbach,  Ver.1.0,  Mar.29,1990 
%                    last test on Mar.26,1990

if (nargin~=3) & (nargin~=4)
  error('Wrong number of input arguments')
end
if nargout~=2
  error('Wrong number of output arguments')
end
if nargin==3, leaveR=1; end
if (max(size(useR)) > 1) | (max(size(leaveR)) > 1)
  error('USER and LEAVER are scalars')
end
if (useR < 1) | (leaveR < 1)
  error('Ranks are positiv:  USER, LEAVER  1')
end
y=y(:);

% some all-data statistics
yav = mean(y);
xav = mean(x);

% pre-allocating
err = zeros(row(x),useR);
y_mr= zeros(leaveR,useR);
cnt = zeros(1,     useR);

press_0=0;
M=row(x);
%fprintf('Method contains %3.0f spectra with %3.0f wavenumbers each\n',M,col(x))
%fprintf('Bidiagonalization aims at rank: useR = %3.0f\n',useR);
for m=1:leaveR:M
  % watch out for odd end conditions
  if (m+leaveR-1) > M, leaveR=M-m+1; end
  ms = m;
  me = m+leaveR-1;
  one = ones(leaveR,1);
  %fprintf([13,'currently deleted no %3.0f - %3.0f'],ms,me)
  % update averages and 'R=0 estimator of PRESS'
  xavup = (M*xav - one'*x(ms:me,:))/(M-leaveR);
  yavup = (M*yav - one'*y(ms:me)  )/(M-leaveR);
  press_0 = press_0 + sum( (y(ms:me)-yavup).*(y(ms:me)-yavup) );
  % update data
  xup = x; xup(ms:me,:) = [];
  xup = xup - ones(M-leaveR,1)*xavup;
  yup = y ; yup(ms:me) = [];
  yup = yup - ones(M-leaveR,1)*yavup;
  % bidiagonalization
  [u,b,w] = lanbi(xup,yup,useR,'noprint');
  y_mr=zeros(leaveR,col(b));
  for r=1:col(b)
   y_mr(:,r)=yavup+((x(ms:me,:)-one*xavup)*w(:,1:r))*inv(b(1:r,1:r))*(u(:,1:r)'*yup);
  end
  if col(y_mr) < useR
    y_mr = [y_mr,diag(y_mr(:,col(b)))*ones(leaveR,useR-col(y_mr))];
  end
  % cnt may be used to control premature stopping
  cnt = cnt + [ones(1,col(b))*leaveR, zeros(1,useR-col(b))];
  err(ms:me,:) = y(ms:me) * ones(1,useR) - y_mr;
end
%fprintf('\n')
% err = matrix of prediction errors: row <=> standard, column <=> rank (1,2..)
% note: rank-zero-prediction (by average value) is not included!
% optional:
%   save c:\matlab\user\pred_err err
%   fprintf('matrix ''err'' of prediction errors (M-by-R) saved\n')
%   fprintf('in MAT:USER:PRED_ERR.MAT; remember Ypred = Ytrue - err !\n')
%disp([' ',7,7,7,7,7,7,7,7,7])
press = sqrt([press_0/M, mean(err.*err)]);
cnt = [M,cnt];
press = [press;cnt];