function [res,lev,lc_res,stud_res,CooksD,lof,crit] = plsout(u,b,v,xav,y,outeR,xnew,ynew)
%PLSOUT PLS outlier detection using various statistical tests.
%
%       [res,lev,lc_res,stud_res,CooksD,Lof,Crit] = plsout(U,B,V,Xav,Y,OuteR)   or
%
%       [res,lev,lc_res,stud_res,CooksD,Lof,Crit] = plsout(U,B,V,Xav,Y,OuteR,Xnew,Ynew)
%
%       Determines various outlier statistics with respect to a PLS regression [U,B,V,Xav,Y]
%       obtained from running PLS2.M. Y is the YSI references, Xav is the mean spectrum, and
%       U*B*V' is the PLS decomposition obtained from the calibration in PLS2.M.
%       OuteR is the PLS rank at which the outlier statistics are determined, 1<OuteR<rank(B).
%       The PLS calibration [U,B,V] should be given at maximum rank because lack-of-fit
%       requires ranks higher than OuteR.
%
%       If Xnew and Ynew are NOT given, then statistics with respect to the calibration data set
%       [U,B,V,Xav] itsself are produced, i.e., the usual calibration outlier statistics.
%       If a new prediction data set [Xnew,Ynew] is given, then the outlier statistics of the
%       new data set [Xnew,Ynew] w.r.t. the old calibration set are produced. (Xnew, Ynew are
%       not mean-centered because OUTLIER internally centers to the old Xav, Yav).
%
%       Oulier statistics are delivered as column vectors:
%        res    = True - Predicted/Fitted [mg/dL]
%        lev    = leverage value (Mahanalobis' distance to Xav in OuTeR-dimensional PLS subspace)
%        lc_res = leverage-corrected residuum (estimates leave-one-out prediction error) [mg/dL]
%        st_res = studentized residuum
%        CooksD = Cook's distance (Mahanalobis's distance of b-vector)
%        LoF    = Lack-of-fit (Euclidian length of spectra outside OuteR-dimensional PLS subspace)
%            In addition, the (1x4)-vector:
%        Crit   = [lcrit,tcrit,Ccrit,Lcrit] suggests critical values for: lev, st_res, CooksD,
%                 and LoF, respectively (see body of program for details of computation).

% R. Marbach, Ver.1.0, 15 Oct 96 (basically a copy of PLS.M by R. Marbach, Ver.1.1, Oct.30,1990)
%                 1.1   8 Jan 96 improved memory requirements
 
K=row(v); % channels
if (nargin==6)
   self=1;
elseif (nargin==8)
   self=0;
   if col(xnew)~=K, error('Wrong number of columns in Xnew.'), end
   ynew=ynew(:);
else
   error('Wrong number of input arguments.')
end

if min(size(xav))~=1
  error('Xav has to be a vector.')
end
xav=xav(:)';
y=y(:);

Rmax=row(b);
if (max(size(outeR)) > 1) | (outeR > Rmax) | (outeR < 1)
      error('Check your input argument "outeR."')
end
fprintf('The lack-of-fit statistic will be based on PLS ranks %2.0f - %2.0f\n', outeR+1, Rmax)

% Divide into two subspaces used/not used in PLS fit
un=u(:,outeR+1:Rmax );  bn=b(outeR+1:Rmax ,outeR+1:Rmax );  vn=v(:,outeR+1:Rmax );
u =u(:,      1:outeR);  b =b(      1:outeR,      1:outeR);  v =v(:,      1:outeR);

% statistics assumming (self==1)
M   = row(u);
yav = mean(y);
res = y-yav - u*(u'*(y-yav));
SSE = norm(res)^2;
sigma = sqrt(SSE/(M-1-outeR));
lev     = sum((u.*u)')';
lc_res  = res./(1-lev);
stud_res= res./sqrt(1-lev)/sigma;
CooksD  = (stud_res.*stud_res).*lev./(1-lev)/(outeR+1);
% b-vector
bvec = v*inv(b)*(u'*y);
% residual squared spectral distance to Xav of calibration spectra
lof  = sum( ((un*bn*vn').^2)' )'/(K-outeR);
pRSD = sum(lof)/(M-outeR-1);  % pooled over all calibration spectra


% --------- definition of critical values w.r.t. calibration -----------
l_crit=2*outeR/M;                         % leverage (2*average value)
t_crit=t_inv(M-1-outeR,0.975);            % StudRes  (two-tailed 5% significant)
C_crit=F_inv(outeR,M-1-outeR,0.001);      % CooksD   (0.1%-move of regr. vector)
L_crit=pRSD*F_inv(K-outeR, ...            % Lof      (alpha=0.05)
      (M-outeR-1)*(K-outeR), 0.95);
crit=[l_crit, t_crit, C_crit, L_crit];

% Outlier statistics for new data set
if self==0
   % center new data w.r.t. old calibration data
   xnew=xnew-ones(row(xnew),1)*xav;
   ynew=ynew-yav;
   res     = ynew - xnew*bvec;
   %   tmp=xnew*v; lev=diag(tmp*inv(b'*b)*tmp');  needs to much computer memory
    [ol,e,or]=svd(b,0); tmp=(xnew*v)*or*diag(diag(e).\1);
   lev     = sum((tmp').^2)';
   lc_res  = res./(1-lev);
   stud_res= res./sqrt(1-lev)/sigma;
   CooksD  = (stud_res.*stud_res).*lev./(1-lev)/(outeR+1);
   lof     = sum( ((xnew*vn*vn').^2)' )'/(K-outeR);
end