function [stat,u,b,v,xav,yav] = pls2(x,y,useR)
%PLS2 Summary statistics of PLS regression.
%
%     STAT= PLS2(X,Y)        or    [STAT,U,B,V,Xav] = PLS2(X,Y)        or
%     STAT= PLS2(X,Y,useR)   or    [STAT,U,B,V,Xav] = PLS2(X,Y, useR)
%
%     Given a matrix X (e.g. spectra) and a vector Y (e.g. glucose), computes summary statistics
%     of the PLS regression of Y versus X, up to a maximum rank. If useR is given, decomposition
%     stops at rank useR. Data X,Y do not need to be mean-centered because PLS2 will mean-center
%     the data internally before performing the PLS decomposition. Y has to be a vector.
%
%                            STAT=[r, SSE, Sig, F, Rsq]
%
%     where, column-wisely, r=PLS ranks, SSE=sum of squared errors, Sig=std of the fit,
%     F=F-test (Ho:regression vector=0), and Rsq=%-variance explained by fit.
%     SSE, Sig, F, and Rsq are functions of the PLS-rank r.
%
%     Optionally, for the purpose of running a subsequent outlier test (using PLSOUT.M), the
%     following can also be delivered: Xav=mean(X), and (X-1*Xav)=U*B*V' the PLS-decomposition
%     obtained on the mean-centered data X-1*mean(X) and Y-mean(Y).

%     R. Marbach, Ver.1.1, 11 Oct 96
%     (basically a copy of PLS.M by R.Marbach, Ver.1.1, Oct.30,1990)

Rmax=min([size(x,1)-1, size(x,2)]);
if (nargin~=2) & (nargin~=3)
   error('Wrong number of input arguments.')
elseif nargin==2
   useR=Rmax;
elseif nargin==3
   if (max(size(useR)) > 1) | (useR > Rmax) | (useR < 1)
      error('Check your input argument "useR."')
   end
end

if min(size(y))~=1
  error('Only "PLS1" type algorithm is implemented => Y has to be a vector')
end
y=y(:);

% some useful statistics
xav = mean(x);         % -> output
yav = mean(y);         % -> output
M     = size(x,1);
TSS   = y'*y;
SSM   = M*yav*yav;
TSSadj= TSS - SSM;

fprintf('bidiagonalizing mean-centered data ..\n')
[u,b,v]= lanbi(x-ones(M,1)*xav, y-yav, useR);         % -> output
PLS_R = size(b,1);                                       % max. PLS rank as determined by LANBI.m
SSE   = zeros(PLS_R,1);
LC_SSE= zeros(PLS_R,1);

fprintf('calculating summary statistics ..\n')
res = y - yav;
lev = zeros(M,1);
for r=1:PLS_R
  res = res - u(:,r)*(u(:,r)'*(y-yav));
  SSE(r)=sum(res.*res);
  lev = lev + u(:,r).*u(:,r);
  LC_SSE(r)=sum(res.*res./(1-lev)./(1-lev));
end

ssr = TSSadj-SSE;
msr = ssr./[1:PLS_R]';
mse = SSE./(M-1-[1:PLS_R]');
Sig = sqrt(mse);
Fval= f([1:PLS_R]',M-1-[1:PLS_R]',msr./mse);
Rsq = 1-SSE./(ssr+SSE);

stat=[ [1:PLS_R]', SSE, Sig, Fval, Rsq];