function y=chi2_inv(v,x)
% CHI2_INV  provides the inverse of the chi-square distribution.
%
%           Y = CHI2_INV(V,X)
%
%           Given values (0)X(1) and V(>0), BETAIINV computes
%           the value of Y satisfying the relation
%                         
%                           1        y   -0.5t   0.5v-1
%           I (v) = ---------------     e       t       dt = x
%            y        0.5v         0 
%                    2    * (0.5v)
%
%           Thus, if X is the value of the lower tail area of the
%           chi-square distribution with V degrees of freedom,
%           Y is the quantile.
%
%           Default accuracy is about six decimal places (see TOL).

%           The inverse chi-square distribution is evaluated by using
%           the algorithm of Best and Roberts (1975), as described in:
%           P.Griffiths, I.D.Hill (eds),'Applied Statistics Algorithms',
%           Royal Statistical Society 1985, Algorithm AS 91, 157-161
%
%           R.Marbach,  Ver.1.0,  Mar.06,1990

if (nargin~=2)
  error('Wrong number of input arguments')
end

if any(size(v)~=size(x))
  error('input arguments don''t match')
end

if any(any(v <= 0))
 error('input parameter(s) out of range: v > 0')
end

% set accuracy to desired level
% if d, say, is the number of accurate decimal places
% desired, the exponent of TOL should be set at -d.
tol=0.5E-6;

% set permissible range of input argument x (probabilitiy)
lowerx = tol;    % 0.000002;
upperx = 1-tol;  % 0.999998;
if any(any(x < lowerx)) | any(any(x > upperx))
 str = ..
  sprintf('input argument(s) out of range: %10.8f  x  %10.8f',lowerx,upperx);
 error(str)
end

% set maximal number of iterations
it1max=10; % newton steps
it2max= 5; % Cornish-Fisher steps

% set constants
aa=0.6931471806;
c1= 0.01;
c2= 0.222222;
c3= 0.32;
c4= 0.4;
c5= 1.24;
c6= 2.2;
c7= 4.67;
c8= 6.66;
c9= 6.73;
c10= 13.32;
c11= 60;
c12= 70;
c13= 84;
c14= 105;
c15= 120;
c16= 127;
c17= 140;
c18= 175;
c19= 210;
c20= 252;
c21= 264;
c22= 294;
c23= 346;
c24= 420;
c25= 462;
c26= 606;
c27= 672;
c28= 707;
c29= 735;
c30= 889;
c31= 932;
c32= 966;
c33= 1141;
c34= 1182;
c35= 1278;
c36= 1740;
c37= 2520;
c38= 5040;

% save input format of x in variable y
[m,n]=size(x);
y = zeros(m,n);

% strung out as column vectors
x = x(:);
v = v(:);

% set constant vector
lngamma=log(gamma(0.5*v));

% allocate temporary variables
%
[m,n]=size(x);
yc = zeros(m,n);
zz = zeros(m,n);
z = zeros(m,n);
c = zeros(m,n);
ch = zeros(m,n);
a = zeros(m,n);
q = zeros(m,n);
p1 = zeros(m,n);
p2 = zeros(m,n);
t = zeros(m,n);

%
% calculate starting approximation for Cornish-Fisher iteration
%
zz=0.5*v;
c=zz-1;

index1  = (v >= (-c5*log(x)));
index1N = ~index1;
if any(index1)
  index2  = index1 &  (v > c3);
  index2N = index1 & ~(v > c3);
  if any(index2)
    z(index2) = inverf(x(index2)); % accurate to about 1e-5
    %
    % starting appr. of Wilson and Hilferty
    p1(index2)= v(index2) .\c2;  
    ch(index2)= v(index2).*(z(index2).*sqrt(p1(index2))+1-p1(index2)) ..
                         .*(z(index2).*sqrt(p1(index2))+1-p1(index2)) ..
                         .*(z(index2).*sqrt(p1(index2))+1-p1(index2));
    %
    % polishing starting appr. for x tending to 1
    index3 = index2 & (ch > (c6*v+6));
    if any(index3)
      ch(index3)=-2*(log(1-x(index3))-c(index3) ..
                                      .*log(0.5*ch(index3))+lngamma(index3));
    end
  end
  if any(index2N)
    ch(index2N)=c4;
    a(index2N)=log(1-x(index2N));
    % Newton-Raphson iteration to find starting appr.
    runindex = index2N;
    for i=1:it1max
      q(runindex)=ch(runindex);
      p1(runindex)=1+ch(runindex).*(c7+ch(runindex));
      p2(runindex)=ch(runindex).*(c9+ch(runindex).*(c8+ch(runindex)));
      t(runindex)=-0.5 +(c7+2*ch(runindex))./ p1(runindex) ..
                   -(c9+ch(runindex).*(c10+3*ch(runindex)))./ p2(runindex);
      ch(runindex)=ch(runindex) -(1 -exp(a(runindex)+lngamma(runindex) ..
                    + 0.5*ch(runindex) +aa*c(runindex)) ..
                    .*p2(runindex)./p1(runindex))./t(runindex);
      %
      % update runindex and watch for break
      runindex = runindex & (abs(q./ch-1) > c1);
      if ~any(runindex), break, end
      if i==it1max
        disp('Warning CHI2_INV: number of Newton iterations exceeds IT1MAX')
      end
    end % Newton loop
  end %index2N
end %index1

indexreadyN = ones(m,n);
if any(index1N)
  ch(index1N)=exp( zz(index1N).\1 .*log(x(index1N).*zz(index1N) ..
                                  .*exp(lngamma(index1N)+aa*zz(index1N))) );
  indexready  = index1N & (ch < tol);
  indexreadyN = ~indexready;
end


%
% Cornish-Fisher iteration to polish starting appr.
%
if any(indexready)
  yc(indexready)=ch(indexready);
end
if any(indexreadyN)
  %
  % Cornish-Fisher iteration
  runindex = indexreadyN;
  for i=1:it2max
    q(runindex)=ch(runindex);
    p1(runindex)=0.5*ch(runindex);
    p2(runindex)=x(runindex)-gamma(zz(runindex),p1(runindex));
    t(runindex)=p2(runindex).*exp(aa*zz(runindex) + lngamma(runindex) + ..
                                  p1(runindex)-c(runindex).*log(ch(runindex)));
    % variable b und s1-s6 lokal!
    b=t(runindex)./ch(runindex);
    a(runindex)=0.5*t(runindex)-b.*c(runindex);
    % local versions of a, c and t
    al=a(runindex);
    cl=c(runindex);
    tl=t(runindex);
    s1=(c19+al.*(c17+al.*(c14+al.*(c13+al.*(c12+c11.*al))))) /c24;
    s2=(c24+al.*(c29+al.*(c32+al.*(c33+c35.*al))))           /c37;
    s3=(c19+al.*(c25+al.*(c28+al.*c31)))                     /c37;
    s4=(c20+al.*(c27+al.*c34) + cl.*(c22+al.*(c30+c36.*al))) /c38;
    s5=(c13+al.*c21 + cl.*(c18+c26.*al))                     /c37;
    s6=(c15 + cl.*(c23+c16.*cl))                             /c38;
    ch(runindex)=ch(runindex) + ..
         tl.*(1+0.5.*tl.*s1-b.*cl.*(s1-b.*(s2-b.*(s3-b.*(s4-b.*(s5-b.*s6))))));
    %
    % update runindex and watch for break
    runindex = runindex & (abs(q./ch-1) > tol);
    if ~any(runindex), break, end
    if i==it2max
      disp('Warning CHI2_INV: number of C-F iterations exceeds IT2MAX')
    end       
  end %Cornish-Fisher
  yc(indexreadyN)=ch(indexreadyN); 
end %indexreadyN

% restore results into required shape
y(:) = yc;
