function y=beta(p,q,x)
%
% BETA provides the values of (1) complete beta function and
%                             (2) incomplete beta ratio function,
%      the latter one with an accuracy of about 6 decimal digits (see TOL).
%
%                         1   p-1      q-1
%      (1) BETA(P,Q)   =     t    (1-t)    dt = BETA(Q,P)
%                         0 
%                        
%                         x   p-1      q-1
%      (2) BETA(P,Q,X) =     t    (1-t)    dt / BETA(P,Q)
%                         0 
%
%          with parameters: p > 0, q > 0 and
%          input arguments: 0  x  1.

%
%      The incomplete beta function is evaluated by using the reduction
%      formula of Sober (1921), see: P.Griffiths, I.D.Hill (eds),'Applied
%      Statistics Algorithms', Royal Statistical Society 1985,
%      Algorithm AS 63, 117-120
%
%      R.Marbach,  Ver.1.0,  Mar.01,1990
%

if (nargin~=2) & (nargin~=3)
  error('Wrong number of input arguments')

elseif nargin==2
 if any(size(p)~=size(q))
   error('input arguments don''t match')
 end
 y = exp( gammacln(p) + gammacln(q) - gammacln(p+q) );

elseif nargin==3
 if any(size(p)~=size(q)) | ...
    any(size(x)~=size(p)) | ...
    any(size(x)~=size(q))
   error('input arguments don''t match')
 end

 if any(any(p <= 0)) | any(any(q <= 0))
  error('input parameter(s) out of range: p > 0, q > 0')
 end

 if any(any(x < 0)) | any(any(x > 1))
  error('input argument(s) out of range: 0  x  1')
 end

 % save input format of x in variable y
 [m,n]=size(x);
 y = zeros(m,n);
 % strung out as column vectors
 xc = x(:);
 pc = p(:);
 qc = q(:);
 [m,n]=size(xc);
 yc = zeros(m,n);

 % betai=0
 index0 = (xc == 0);
 if any(index0), yc(index0) = 0; end
 % betai=1
 index1 = (xc == 1);
 if any(index1), yc(index1) = 1; end

 indexwork = ~(index0 | index1);
 if any(indexwork)
   xtmp=xc(indexwork);
   ptmp=pc(indexwork);
   qtmp=qc(indexwork);

   % set accuracy to desired level
   tol=1E-7;

   % allocate temporary variables
   [m,n]=size(xtmp);
   xx = zeros(m,n);
   cx = zeros(m,n);
   rx = zeros(m,n);
   pp = zeros(m,n);
   qq = zeros(m,n);
   psq = zeros(m,n);
   temp = zeros(m,n);
   ns = zeros(m,n);

   % change tail if necessary and determine s
   psq = ptmp + qtmp;
   cx = 1 - xtmp;
   ifindex = (ptmp >=(psq.*xtmp));
   if any(ifindex)
     xx(ifindex)=xtmp(ifindex);
     pp(ifindex)=ptmp(ifindex);
     qq(ifindex)=qtmp(ifindex);
   end
   if any(~ifindex)
     xx(~ifindex)=cx(~ifindex);
     cx(~ifindex)=xtmp(~ifindex);
     pp(~ifindex)=qtmp(~ifindex);
     qq(~ifindex)=ptmp(~ifindex);
   end
   term=ones(m,n);
   ai=ones(m,n);
   ytmp=ones(m,n);
   ns=fix(qq+cx.*psq);

   % use reduction formula of sober
   rx=xx./cx;
   temp=qq-ai;
   if2index=(ns == 0);
   if any(if2index)
     rx(if2index)=xx(if2index);
   end
   term=term .* temp .* rx ./ (pp+ai);
   ytmp = ytmp + term;
   temp=abs(term);
   while ~all( (temp<=tol) & (temp<=(tol.*ytmp)) )
     ai=ai+1;
     ns=ns-1;
     if3index= (ns >= 0);
     if any(if3index)
       temp(if3index)=qq(if3index)-ai(if3index);
       if22index=(ns(if3index) == 0);
       if any(if22index)
         rx2=rx(if3index);
         xx2=xx(if3index);
         rx2(if22index)=xx2(if22index);
         rx(if3index)=rx2;
         xx(if3index)=xx2;
       end
       term(if3index)=term(if3index).*temp(if3index).*rx(if3index) ...
                      ./ (pp(if3index)+ai(if3index));
       ytmp(if3index)=ytmp(if3index)+term(if3index);
       temp(if3index)=abs(term(if3index));
     end
     if any(~if3index)
       temp(~if3index)=psq(~if3index);
       psq(~if3index)=psq(~if3index)+1;
       term(~if3index)=term(~if3index).*temp(~if3index).*rx(~if3index) ...
                       ./ (pp(~if3index)+ai(~if3index));
       ytmp(~if3index) = ytmp(~if3index) + term(~if3index);
       temp(~if3index) = abs(term(~if3index));       
     end
   end %while

   % calculate result
   lnbeta=log(beta(p(indexwork),q(indexwork)));
   lnbeta=lnbeta(:);
   ytmp=ytmp.*exp( pp.*log(xx)+(qq-1).*log(cx)-lnbeta )./pp;
   if any(~ifindex)
     ytmp(~ifindex)=1-ytmp(~ifindex);
   end
 end %indexwork
 
 yc(indexwork)=ytmp;
 y(:) = yc;
end